/*
 * cSubnetTree.cpp
 * 
 * Class implementation for a subnet tree for pbcast.
 *
 * Authors:  Vicky & Ted
 */
#include "cSubnetTree.h"
#include "Endpoint/cEndpoint.h"
#include <math.h>
#include <assert.h>
#include <stdlib.h>

#ifndef NULL
	#define NULL 0
#endif

/*
 * cSubnetTree::cSubnetTree()
 *
 * Purpose:	Creates a new subnet tree.
 * IN:		initialCap	-> The starting size of the queue.
 *			capInc		-> How much to grow the queue by every time.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The queue is set up.
 * Throws:	-
 * Return:	-
 */
cSubnetTree::cSubnetTree(unsigned int initialCap, unsigned int capInc)
{
	mSize			= initialCap;
	mInsertPtr		= 0;
	mCapIncrement   = capInc;

	// Alloc queue space.
	mQueue = new SubnetInfo[mSize];
	if(!mQueue)
	{
		//throw cQueueException("<cSortedMsgQueue::cSortedMsgQueue>: Unable to alloc new queue.");
	}
	for(unsigned int i = 0; i < mSize; i++)
	{
		mQueue[i].subnet = NULL;
		mQueue[i].lastGossipOnSubnet = NULL;
	}
}

/*
 * cSubnetTree::~cSubnetTree()
 *
 * Purpose:	Destroy the queue.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The queue is cleaned up.
 * Return:	-
 */
cSubnetTree::~cSubnetTree()
{
	if(mQueue) 
	{ 
		Clear();
		delete mQueue; 
	}
}

/*
 * cSubnetTree::GetRandomSubnet()
 *
 * Purpose:	Gets a random subnet endpoint from the tree.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The queue shrinks by at least one.
 * Return:	true if success, else fail.
 */
cEndpoint* cSubnetTree::GetRandomSubnet()
{
	if(GetNumElements())
	{
		unsigned int chosen = (unsigned)rand();
		chosen = chosen % GetNumElements();
		return mQueue[chosen].subnet;
	}
	else
	{
		return NULL;
	}
}

/*
 * cSubnetTree::GetRandomSubnet()
 *
 * Purpose:	Gets a random subnet from the tree.
 * IN:		info		-> Address of an existing info struct
 *			rootIndex	-> Address of a int to hold the rootindex
 * OUT:		-
 * Cond:	-
 * PostCnd:	The queue shrinks by at least one.
 * Return:	true if success, else fail.
 */
bool	cSubnetTree::GetRandomSubnet(SubnetInfo* info, unsigned int* rootIndex)
{
	if(GetNumElements())
	{
		unsigned int chosen = (unsigned)rand();
		chosen = chosen % GetNumElements();
		*rootIndex = chosen;
		*info	   = mQueue[chosen];
		return true;
	}
	else
	{
		return false;
	}
}

/*
 * cSubnetTree::Delete()
 *
 * Purpose:	Deletes the given subnet from the subnet tree.
 * IN:		subnet	-> The subnet to delete if it is present.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The queue shrinks by at least one.
 * Return:	true if success, else false.
 */
bool cSubnetTree::Delete(cEndpoint* subnet)
{
	if(mInsertPtr)
	{
		int	location = _BinSearch(subnet, 0, mInsertPtr-1);
		if(location == -1)
		{
			return false;	// not present
		}
		if(mQueue[location].subnet)
		{
			mQueue[location].subnet->Release();
		}
		if(mQueue[location].lastGossipOnSubnet)
		{
			mQueue[location].lastGossipOnSubnet->Release();
		}

		// Swap elements over to fill hole
		location = location + 1;
		while(location < (int)mInsertPtr)
		{
			mQueue[location-1] = mQueue[location];
			location++;
		}
		mInsertPtr--;
		mQueue[mInsertPtr].subnet = NULL;
		mQueue[mInsertPtr].lastGossipOnSubnet = NULL;
		return true;
	}
	return false;
}

/*
 * cSubnetTree::UpdateGossip()
 *
 * Purpose:	Updates with the info for the last dude i gossiped to on the given subnet.
 * IN:		gossipPartner	-> An endpoint on a subnet.
 * OUT:		-
 * Cond:	The subnet better be in there!
 * PostCnd:	The SubnetInfo.lastGossipOnSubnet param is updated if subnet is found.
 * Return:	the index of the given child.
 */ 
bool cSubnetTree::UpdateGossip(cEndpoint* gossipPartner)
{
	int loc;

	if(mInsertPtr)
	{
		loc = _BinSearch(gossipPartner, 0, mInsertPtr-1);
		if(loc == -1)
		{
			return false;
		}
		gossipPartner->AddRef();
		if(mQueue[loc].lastGossipOnSubnet)
		{
			mQueue[loc].lastGossipOnSubnet->Release();
		}
		mQueue[loc].lastGossipOnSubnet = gossipPartner;
		return true;
	}
	return false;
}

/*
 * cSubnetTree::GetChild()
 *
 * Purpose:	Returns the child of the given index, assuming tree of given degree
 * IN:		rootIndex		-> the root of the tree
 *			currentIndex	-> the index of the node whose child want to find.
 *			childNum		-> The number of the child to find < 1...fanout >
 * OUT:		-
 * Cond:	-
 * PostCnd:	The index is returned for the given child.
 * Return:	the index of the given child or -1 if there is no child.
 */ 
int cSubnetTree::GetChild(unsigned int rootIndex, unsigned int currentIndex, unsigned int childNum, unsigned int fanout)
{
	unsigned int result;
	if(currentIndex < rootIndex)
	{
		currentIndex += mInsertPtr;		// Increase by size for wraparound.
	}
	currentIndex -= rootIndex;					// Subtract off the root.
	result = fanout*currentIndex + childNum;	// Compute child.
	if(result >= mInsertPtr)
	{
		return -1;	// Exceeded the tree.
	}
	result = (result+rootIndex) % mInsertPtr;
	return result;
}

/*
 * cSubnetTree::GetHeight()
 *
 * Purpose:	Returns the expected height of the tree based on the given fanout.
 * IN:		fanout		-> The degree of the tree.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The queue is grown by at least one, at most mCapIncrement elements.
 * Return:	true if success, else false.
 */ 
unsigned int cSubnetTree::GetHeight(unsigned int fanout)
{
	double retVal;
	retVal = log10(mInsertPtr)/log10(fanout);
	return (unsigned int)retVal;
}

/*
 * cSubnetTree::_Grow()
 *
 * Purpose:	Attempts to grow the queue by at least one, at most mCapIncrement
 * IN:		-
 * OUT:		-
 * Cond:	The queue is full!
 * PostCnd:	The queue is grown by at least one, at most mCapIncrement elements.
 * Return:	true if success, else false.
 */ 
bool cSubnetTree::_Grow()
{
	if(mInsertPtr == mSize)
	{
		// We need to expand the queue.
		SubnetInfo* newQ = new SubnetInfo[mSize+mCapIncrement];
		if(!newQ)
		{
			return false;
		}

		// The following may be faster using memcpy()...
		for(unsigned int i = 0; i < mSize; i++)
		{
			newQ[i] = mQueue[i];
		}
		for(i = mSize; i < mSize+mCapIncrement; i++)
		{
			newQ[i].subnet = NULL;
			newQ[i].lastGossipOnSubnet = NULL;
		}
		mSize = mSize + mCapIncrement;
		delete mQueue;
		mQueue = newQ;
	}
 	return true;
}

/*
 * cSubnetTree::Insert()
 *
 * Purpose:	Attempts to insert the subnet, but only if it is not in the queue.
 * IN:		subnet		-> The subnet to insert
 * OUT:		-
 * Cond:	The subnet is not in the queue.
 * PostCnd:	The subnet is added to the queue.
 * Return:	true if success, else false.
 */ 
bool cSubnetTree::Insert(cEndpoint* subnet)
{
	bool	retVal;
	int		searchResult;

	// See if we need to replace.
	if(mInsertPtr)
	{
		searchResult = _BinSearch(subnet, 0, mInsertPtr-1);
		if(searchResult != -1)
		{
			return true;	// It was already in there, but that is success.
		}
	}

	// Inserting brand new element!
	if(mInsertPtr == mSize)
	{
		if(!_Grow())
		{
			return false;
		}
	}

	mQueue[mInsertPtr].subnet = subnet;
	mQueue[mInsertPtr].subnet->AddRef();
	mQueue[mInsertPtr].lastGossipOnSubnet = NULL;

	// Sort it
	_InsertSort();
	mInsertPtr = mInsertPtr+1;	// Not a circular queue!
	retVal = true;

	return retVal;
}

/*
 * cSubnetTree::_InsertSort()
 *
 * Purpose:	Keeps the sorted property of the queue, assuming new element is placed
 *			at mInsertPtr
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if a should be delivered before b, else false.
 */ 
bool cSubnetTree::_InsertSort()
{
	unsigned int	current, prev;
	SubnetInfo		temp;

	current = prev = mInsertPtr;
	while(current != 0)
	{
		prev = prev-1;		// No wrap around!

		// Make sure that prev is not a hole!
		if(mQueue[prev].subnet)
		{
			if(!_Before(mQueue[current].subnet, mQueue[prev].subnet))
			{
				return true;		// No need to swap anymore
			}
		}

		// Swap (into a hole if one exists)
		temp = mQueue[current];
		mQueue[current] = mQueue[prev];
		mQueue[prev] = temp;

		current = prev;
	}
	return true;
}

/*
 * cSubnetTree::_Before()
 *
 * Purpose:	Returns true if a is before b
 * IN:		a, b	-> msgs to compare.
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if a is before b, else false.
 */ 
bool cSubnetTree::_Before(cEndpoint* a, cEndpoint* b)
{
	return a->SubnetLessThan(b);
}

/*
 * cSubnetTree::_Equal()
 *
 * Purpose:	Returns true if a refers to same subnet as b
 * IN:		a, b	-> msgs to compare.
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if a is the same as b
 */ 
bool cSubnetTree::_Equal(cEndpoint* a, cEndpoint* b)
{
	return a->IsSameSubnet(b);
}

/*
 * cSubnetTree::Find()
 *
 * Purpose:	Searches for the element and returns its index.
 * IN:		subnet		-> The subnet to look for.
 * OUT:		-
 * Cond:	
 * PostCnd:	
 * Return:	-1 if not in there, else index.
 */
int cSubnetTree::Find(cEndpoint* subnet)
{
	int retVal;
	if(!mInsertPtr)
	{
		return false;
	}
	retVal = _BinSearch(subnet, 0, mInsertPtr-1);

	return retVal;
}

/*
 * cSubnetTree::_BinSearch()
 *
 * Purpose:	Performs binary search, looking for a match.
 * NOTE:	The input left/right can NOT be wrap around values.
 * IN:		subnet		-> The subnet to look for.
 *			left		-> The left bounds of the array (start w/ 0)
 *			right		-> The right bounds of the array (start w/ insert ptr-1)
 * OUT:		-
 * Cond:	The Subnet is in the queue.
 * PostCnd:	-
 * Return:	-1 if not found, else the index in mQueue of the SubnetInfo
 */
int cSubnetTree::_BinSearch(cEndpoint* subnet, int left, int right)
{
	unsigned int pivot;

	while(1)
	{
		if(left > right)
		{
			return -1;
		}
		else
		{
			pivot = (right-left) / 2;	// Just how much pivot advances from left, actually.
		}
		pivot = (left + pivot);	

		// See where to search next
		if(_Before(subnet, mQueue[pivot].subnet))
		{
			right = pivot-1;
		}
		else if( _Equal(subnet, mQueue[pivot].subnet) )
		{
			// Found it!
			return pivot;
		}
		else
		{
			// Search from pivot+1 to right
			left = pivot+1;
		}
	}
}

/*
 * cSubnetTree::Clear()
 *
 * Purpose:	Clears out the queue.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The queue is reset and cleared.
 * Return:	true if success, false otherwise.
 */
bool cSubnetTree::Clear()
{
	for(unsigned int i = 0; i < mSize; i++)
	{
		if(mQueue[i].subnet)
		{
			mQueue[i].subnet->Release();
		}
		if(mQueue[i].lastGossipOnSubnet)
		{
			mQueue[i].lastGossipOnSubnet->Release();
		}
	}
	mInsertPtr   = 0;
	return true;
}

// SUBNET-TREE ITERATOR
bool cSubnetTreeIterator::Done()		{ return mIndex == mTree->mInsertPtr;  }
cObject* cSubnetTreeIterator::GetData() { return mTree->mQueue[mIndex].subnet; }
bool	 cSubnetTreeIterator::GetNext() { mIndex++; return true; }
bool	 cSubnetTreeIterator::DeleteCurrent() { return false; }